/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package jdbc;

/**
 * With this database class it is possible to set the data for main and sub-report with an external 
 * Result Set instead of using a connection to a database created by i-net Clear Reports.
 *
 * To do this you have to 
 * 1) create a Data Source Configuration with a user-defined driver for this database class
 * 2) set the name of the Data Source Configuration in the parameter datasource at runtime
 * 
 * Create a Data Source Configuration using the Data Source Manager:
 * - select "user-defined driver"
 * - set a name for the Data Source Configuration, e.g. "MainAndSubDataWithExternalResultSet"
 * - set this class "samples.jdbc.Main_and_Subreport_Data_WithExternalResultSet" as Database Class for the Data Source Configuration
 * 
 * In the report url you have to set the name of the Data Source Configuration in the datasource parameter:
 * ...datasource=MainAndSubDataWithExternalResultSet...
 */

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.Statement;

import com.inet.report.Engine;
import com.inet.report.ReportException;
import com.inet.report.database.TableData;
import com.inet.report.database.fetch.DataCollector;
import com.inet.report.database.fetch.FetchTables;

/**
 * shows how you can distinguish between main and sub report and provide different data for each with your own Database
 * class. See {@link DatabaseClassUsableInDesigner} for instructions on how to add this class as a datasource from
 * within i-net Designer.
 */
public class MainAndSubreportDataWithExternalResultSet extends com.inet.report.Database {
    // Sample for creating the resultset by executing a SQL Query with a JDBC driver from i-net software.
    // This method will be called once for the main report and once for each sub report.
    /**
     * {@inheritDoc}
     */
    @Override
    public void fetchData( Engine engine, FetchTables fetchTables, DataCollector dataCollector ) throws ReportException {
        ResultSet rs = null;
        Statement st = null;

        /* Methods to see which report or subreport is working with this engine
        engine.getReportTitle();
        engine.getPrompts();
        ...
        */

        try {
            String url = "jdbc:inetdae:<serverName>?database=<dbName>"; // your JDBC connection URL
            Connection con = DriverManager.getConnection( url, "<userName>", "<pwd>" ); // your connection info
            st = con.createStatement();
            // The column name(s) need to be the same as in the report (rpt file)
            // If this result set have different column names as the report then you can use alias names.
            Engine parentEngine = engine.getParent();
            if( parentEngine == null ) {
                // data for the main report 
                System.out.println( "Setting data for the main report" );
                rs = st.executeQuery( "Select * From ..." );
            } else {
                // data for the sub report
                // here you can use the engine API to determine the subreport, e.g. Engine.getReportTitle()
                System.out.println( "Setting data for the sub report" );
                rs = st.executeQuery( "Select * From ..." );
            }
            dataCollector.addUnjoinedData( fetchTables.getTableSources().get( 0 ), new TableData( rs ) );
        } catch( Exception e ) {
            e.printStackTrace();
            //...
        } finally {
            try {
                st.close();
                st = null;
            } catch( Exception e ) {
                e.printStackTrace();
                //...
            }
            try {
                rs.close();
                rs = null;
            } catch( Exception e ) {
                e.printStackTrace();
                //...
            }
        }
    }
}
